bl_info = {
    "name": "Pixel Plow render farm service",
    "author": "Pixel Plow",
    "category": "Render",
    "description": "Streamlines submitting Blender scenes to the Pixel Plow render farm service.",
    "wiki_url": "https://www.pixelplow.net/support/",
    "blender": (2, 80, 0),
    "version": (1, 0),
}


import os
import platform
import bpy
import json
from traceback import format_exc


class RenderPixelPlow(bpy.types.Operator):
    """Submit the scene to the Pixel Plow render farm agent"""
    bl_idname = "render.pixel_plow"
    bl_label = "Render on Pixel Plow"
    bl_options = {'REGISTER'}

    def execute(self, context):
        if not context.preferences.filepaths.use_relative_paths:
            def draw_absolute_path_error_popup(self, context):
                self.layout.label(text='Please select relative file paths and resubmit.')
                self.layout.prop(context.preferences.filepaths, 'use_relative_paths')

            context.window_manager.popup_menu(draw_absolute_path_error_popup, title='Pixel Plow', icon='ERROR')
            return {'FINISHED'}

        if bpy.data.is_dirty:
            bpy.ops.wm.save_mainfile()

        is_mac = platform.system() == 'Darwin' or os.sep == '/'
        is_pc = not is_mac

        if is_pc:
            watch_file_pathname = os.path.join(os.environ['APPDATA'], 'PixelPlow', 'watchJob.json')
        else:
            watch_file_pathname = os.path.expanduser('~/Library/PixelPlow/watchJob.json')

        if os.path.exists(watch_file_pathname):
            def draw_watch_file_exists_error_popup(self, context):
                self.layout.label(text='Please start the Pixel Plow agent and complete or cancel the existing job submission.')

            context.window_manager.popup_menu(draw_watch_file_exists_error_popup, title='Pixel Plow', icon='ERROR')
            return {'FINISHED'}

        job_data = {
            'renderapp': 'Blender 2.8',
            'scenefile': bpy.data.filepath,
            'outputformat': context.scene.render.file_extension.lstrip('.'),
        }

        if context.scene.frame_step == 1:
            job_data['framelist'] = '{}-{}'.format(context.scene.frame_start, context.scene.frame_end)
        else:
            job_data['framelist'] = '{} to {} by {}'.format(
                context.scene.frame_start,
                context.scene.frame_end,
                context.scene.frame_step
            )

        output_basename = bpy.path.basename(context.scene.render.filepath).replace('#', '')
        if output_basename:
            job_data['outputname'] = output_basename

        try:
            with open(watch_file_pathname, 'wt', encoding='utf-8') as watch_file_handle:
                json.dump(job_data, watch_file_handle, indent='\t', sort_keys=True)
        except:
            def draw_write_error_popup(self, context):
                self.layout.label(text='Could not write the watch file for the agent to pick up.')
                print(format_exc())

            context.window_manager.popup_menu(draw_write_error_popup, title='Pixel Plow', icon='ERROR')

        return {'FINISHED'}


def render_menu(self, context):
    self.layout.separator()
    self.layout.operator(RenderPixelPlow.bl_idname)


def register():
    bpy.utils.register_class(RenderPixelPlow)
    bpy.types.TOPBAR_MT_render.append(render_menu)


def unregister():
    bpy.utils.unregister_class(RenderPixelPlow)
    bpy.types.TOPBAR_MT_render.remove(render_menu)


if __name__ == "__main__":
    register()
